/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains routines for the loading of generic font
 *	(GF) files.
 */

#include "types.h"
#include "font.h"
#include "gf.h"
#include "tracedef.h"

#include "arith.p"
#include "raster.p"
#include "fileio.p"
#include "fio.p"
#include "font.p"
#include "heap.p"

#define case_1(k)   case k:
#define case_3(k)   case k: case k+1: case k+2:
#define case_4(k)   case k: case k+1: case k+2: case k+3:
#define case_16(k)  case_4(k) case_4(k+4) case_4(k+8) case_4(k+12)
#define case_64(k)  case_16(k) case_16(k+16) case_16(k+32) case_16(k+48)
#define case_165(k) case_64(k) case_64(k+64) case_16(k+128) case_16(k+144) case_4(k+160) case k+164:

/*
 *	Routine Load_GF reads in the information in a generic font file
 *	and converts it to character raster and dimension information.
 */

int Load_GF (Font_Dir, Resolution, Font_Ptr)
char *Font_Dir;
struct Ratio *Resolution;
struct Font_Definition *Font_Ptr;
{
	auto   pointer GF_File, GF_Heap;
	auto   int Index;
	static unsigned long Addr, Char_Code;
	extern pointer Open_GF_File();
	extern int Process_GF_Character(), Process_GF_Preamble(), Process_GF_Postamble();
	msgcode DVIOUT_BADGFFILE, DVIOUT_NOGFFILE, DVIOUT_UNKGFFORMAT,
		DVIOUT_DUPLCHAR;
/*
 *	Construct default file name; open the GF file; initialize array
 *	indicating which characters have been defined; initialize the
 *	heap used to sort the addresses of character positions within
 *	the file:
 */
	if ((GF_File = Open_GF_File (Font_Dir, Font_Ptr->Name_Ptr, Resolution, Font_Ptr->Magnification)) == 0) {
		Message (DVIOUT_NOGFFILE, Font_Ptr->Name_Ptr, 1, Font_Ptr->Magnification, 0);
		return (0);
	}
	for (Index = 0; Index < CHARS_PER_FONT; Index++) {
		Char_Addr[Index] = 0;
		Char_Mask[Index] = 0;
	}
	GF_Heap = Initialize_Heap_M (CHARS_PER_FONT, 1);
/*
 *	Process the file's preamble and postamble:
 */
	if (Process_GF_Preamble (Font_Ptr, GF_File) == 0)
		goto Bad_File;
	if (Process_GF_Postamble (Font_Ptr, GF_Heap, GF_File) == 0)
		goto Bad_File;
/*
 *	Check the file format:
 */
	if ((Font_Ptr->Pixel_Id & ID_M_FORMAT) != GF_FORMAT_A) {
		Message (DVIOUT_UNKGFFORMAT, Font_Ptr->Name_Ptr, 1, Font_Ptr->Magnification, 0);
		goto Bad_File_1;
	}
/*
 *	Process each character in the font:
 */
	while (Extract_Heap_Entry_M (GF_Heap, &Addr, &Char_Code) != 0) {
		if (Char_Mask[Char_Code] != 0)
			Message (DVIOUT_DUPLCHAR, Char_Code, 0, Font_Ptr->Name_Ptr, 1);
		else if (Set_File_Position_M (Addr, GF_File) != Addr ||
			 Process_GF_Character (Font_Ptr, GF_File) == 0)
			goto Bad_File;
	}
	Dissolve_Heap_M (GF_Heap);
/*
 *	Check for undefined characters:
 */
	for (Index = 0; Index < CHARS_PER_FONT; Index++)
	if (Font_Ptr->Font_Directory[Index] != 0 && Char_Mask[Index] == 0) {
		Font_Ptr->Font_Directory[Index] = 0;
		Undef_Char_Count++;
	}
	Undef_Char_Count += Check_Undefined_Char_M (Font_Ptr, CHARS_PER_FONT, Font_Ptr->Char_Dir_Count);
/*
 *	Set the coding scheme, family name and face type to null:
 */
	Font_Ptr->Font_Coding[0] = '\0';
	Font_Ptr->Font_Family[0] = '\0';
	Font_Ptr->Font_Face = 0;
	Close_File_M (GF_File);
	return (1);
Bad_File:
	Message (DVIOUT_BADGFFILE, Font_Ptr->Name_Ptr, 1, Font_Ptr->Magnification, 0);
Bad_File_1:
	Close_File_M (GF_File);
	return (0);
}

pointer Open_GF_File (Font_Dir, Font_Name, Resolution, Magnification)
char *Font_Dir, *Font_Name;
struct Ratio *Resolution;
unsigned long Magnification;
{
	static char Default_File[80], *Template = "%s.%dgf";

	sprintf (Default_File, Template, Font_Name,
		 Get_Font_Mag_M (1, Resolution, XN_Div_D_R_M (Magnification, Resolution->Numerator,
			         1000 * Resolution->Denominator)));
	if (trace_pixel)
		printf ("Name of the GF file is \"%s\"\n", Default_File);
	return (Open_File_M (Default_File, Font_Dir, "r", 0));
}

int Process_GF_Preamble (Font_Ptr, GF_File)
struct Font_Definition *Font_Ptr;
pointer GF_File;
{
	Set_File_Position_M (0, GF_File);
	if (Read_Character_M (GF_File) != PRE)
		return (0);
	Font_Ptr->Pixel_Id = (ID_GF << ID_V_TYPE) | (Read_Unsigned_Int_M (1, GF_File) & ID_M_FORMAT);
	Skip (Read_Unsigned_Int_M (1, GF_File), GF_File);
	if (trace_pixel)
		printf ("  GF file ID is %lu\n", Font_Ptr->Pixel_Id & ID_M_FORMAT);
	return ((File_At_EOF_M (GF_File) == 0) ? 1 : 0);
}

/*
 *	Routine Process_GF_Postamble reads in the postamble of the
 *	GF file and stores the information found for later use,
 *	checks various values against those found in the Preamble,
 *	and reads in the partial character definitions. The addresses
 *	of the character BOC's, along with their character codes,
 *	are stored in a heap, ordered by file address. Since we have
 *	no assurance that the character descriptions will be in the
 *	same order as they are in the postamble, or by their character
 *	codes, this seems a reasonable way to ensure that processing
 *	of the character descriptions involves the least amount of i/o
 *	on the GF file.
 */

int Process_GF_Postamble (Font_Ptr, GF_Heap, GF_File)
struct Font_Definition *Font_Ptr;
pointer GF_Heap, GF_File;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   unsigned long Addr;
	auto   long Temp, Width, Dx, Dy;
	auto   int Count;
	auto   unsigned char Command, Char_Code;
	extern char *Mem_Alloc();
	extern unsigned long Find_GF_Postamble();
	msgcode DVIOUT_DUPLCHAR;
/*
 *	Locate the Postamble; do some error checking:
 */
	if ((Addr = Find_GF_Postamble (Font_Ptr, GF_File)) == 0 ||
	    Set_File_Position_M (Addr, GF_File) != Addr ||
	    Read_Character_M (GF_File) != POST || File_At_EOF_M (GF_File) != 0)
		return (0);
/*
 *	Read in the design size, checksum; skip hppp, vppp, min_m,
 *	max_m, min_n, max_n:
 */
	Read_Unsigned_Int_M (4, GF_File);
	Font_Ptr->Font_Design_Size = Read_Unsigned_Int_M (4, GF_File);
	Font_Ptr->Checksum = Read_Unsigned_Int_M (4, GF_File);
	if (trace_pixel) {
		printf ("  GF file Design Size is %lu\n", Font_Ptr->Font_Design_Size);
		printf ("  GF file Checksum is %lu\n", Font_Ptr->Checksum);
	}
	for (Count = 6; Count > 0; Count--) {
		Temp = Read_Signed_Int_M (4, GF_File);
		if (trace_pixel)
			printf ("  Additional POST parameter: %ld\n", Temp);
	}
/*
 *	Read in all of the character locators; store in heap:
 */
	while ((Command = Read_Character_M (GF_File)) != POST_POST && File_At_EOF_M (GF_File) == 0)
	if (Command == NO_OP)
		continue;
	else if (Command != CHAR_LOC && Command != CHAR_LOC0) {
		if (trace_pixel)
			printf ("  Bad POST command byte: %u\n", Command);
		break;
	} else {
		Char_Code = Read_Character_M (GF_File);
		if (Command == CHAR_LOC) {
			Dx = Read_Unsigned_Int_M (4, GF_File);
			Dy = Read_Unsigned_Int_M (4, GF_File);
		} else {
			Dx = Read_Unsigned_Int_M (1, GF_File);
			Dy = 0;
		}
		Width = Read_Signed_Int_M (4, GF_File);
		Char_Addr[Char_Code] = Addr = Read_Unsigned_Int_M (4, GF_File);
		if (trace_pixel) {
			printf ("  Character code (modulo 256): %lu\n", Char_Code);
			printf ("    Address of character: %08lX\n", Addr);
			printf ("    Width in scaled points: %ld\n", Width);
			printf ("    Escapement (1/65536 pixel) is %ld by %ld\n", Dx, Dy);
		}
		if (Char_Mask[Char_Code] != 0)
			Message (DVIOUT_DUPLCHAR, Char_Code, 0, Font_Ptr->Name_Ptr, 1);
		else if (Font_Ptr->Font_Directory[Char_Code] != 0) {
			GF_Char_Info[Char_Code].Char_Width = Width;
			GF_Char_Info[Char_Code].H_Esc = (Dx >= 0) ? (Dx + 32) >> 6 : (Dx - 32) >> 6;
			GF_Char_Info[Char_Code].V_Esc = (Dy >= 0) ? (Dy + 32) >> 6 : (Dy - 32) >> 6;
			if (Addr == 0xFFFFFFFF) {	/* Probably an 'invisible' font */
				Char_Ptr = (struct Char_Definition *) Mem_Alloc (sizeof (struct Char_Definition));
				Char_Ptr->Driver_Id = (unsigned long) Font_Ptr->Font_Directory[Char_Code];
				Char_Ptr->DVI_Width = Width;
				Char_Ptr->H_Escapement = GF_Char_Info[Char_Code].H_Esc;
				Char_Ptr->V_Escapement = GF_Char_Info[Char_Code].V_Esc;
				Char_Ptr->Character_Code = Char_Code;
				Char_Ptr->Pixel_Width = 0;
				Char_Ptr->Pixel_Height = 0;
				Char_Ptr->X_Origin = 0;
				Char_Ptr->Y_Origin = 0;
				Char_Ptr->Pixel_Array = 0;
				Font_Ptr->Font_Directory[Char_Code] = Char_Ptr;
				Char_Mask[Char_Code] = 1;
			} else
				Insert_Heap_Entry_M (GF_Heap, Addr, Char_Code);
		}
	}
	return ((Command == POST_POST) ? 1 : 0);
}

/*
 *	Routine Find_GF_Postamble locates the Postamble of the GF
 *	file. This is done by setting to the end of file and reading
 *	backwards until the postamble address is found. It also checks
 *	the GF file id byte to see if it matches the value found in
 *	the preamble.
 */

unsigned long Find_GF_Postamble (Font_Ptr, GF_File)
struct Font_Definition *Font_Ptr;
pointer GF_File;
{
	auto   unsigned long Postamble_Address;
	auto   unsigned int Sig_Count;
	auto   unsigned char Value;

	Set_EOF_M (GF_File);
	Sig_Count = 0;
	while ((Value = Read_Character_Reverse_M (GF_File)) == SIGNATURE)
		Sig_Count++;
	if (Value != (Font_Ptr->Pixel_Id & ID_M_FORMAT))
		return (0);
	Postamble_Address = Read_Unsigned_Int_Reverse_M (4, GF_File);
	if (trace_pixel) {
		printf ("  Number of %d's found: %u\n", SIGNATURE, Sig_Count);
		printf ("  Postamble address is %08lX\n", Postamble_Address);
	}
	return (Postamble_Address);
}

/*
 *	Routine Process_GF_Character reads through the character description
 *	for one character. We do not specifically track the variables 'm'
 *	and 'n' (see GFtype program documentation), but rather keep track
 *	of the address of the current row and the address of the current byte
 *	within that row. All operations are on an unpacked raster array.
 */

int Process_GF_Character (Font_Ptr, GF_File)
struct Font_Definition *Font_Ptr;
pointer GF_File;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   unsigned char *Raster_Ptr, *Row_Ptr, *Cur_Ptr;
	auto   unsigned long Temp, Width, Char_Code;
	auto   int Index;
	auto   unsigned char Command, Paint_Switch;
	static long Char_Bounds[4];
	static unsigned short Wid, Hgt;
	static short X0, Y0;
	extern char *Mem_Alloc();
/*
 *	Read through XXX, YYY and NO_OP commands until the BOC is
 *	seen:
 */
	while ((Command = Read_Character_M (GF_File)) != BOC && Command != BOC1 &&
			File_At_EOF_M (GF_File) == 0)
	if (Command == NO_OP)
		continue;
	else if (Command >= XXX1 && Command <= XXX1+3)
		Skip (Read_Unsigned_Int_M (Command-XXX1+1, GF_File), GF_File);
	else if (Command == YYY)
		Read_Unsigned_Int_M (4, GF_File);
	else break;
/*
 *	Read in the BOC information:
 */
	if (Command == BOC) {
		Char_Code = Read_Unsigned_Int_M (4, GF_File) & 0xFF;
		Temp = Read_Unsigned_Int_M (4, GF_File);
		for (Index = 0; Index < 4; Index++)
			Char_Bounds[Index] = Read_Signed_Int_M (4, GF_File);
	} else if (Command == BOC1) {
		Char_Code = Read_Unsigned_Int_M (1, GF_File);
		Temp = 0xFFFFFFFF;
		for (Index = 0; Index < 4; Index++)
			Char_Bounds[Index] = Read_Unsigned_Int_M (1, GF_File);
		Char_Bounds[0] = Char_Bounds[1] - Char_Bounds[0];
		Char_Bounds[2] = Char_Bounds[3] - Char_Bounds[2];
	} else {
		if (trace_pixel)
			printf ("    Unexpected command while scanning for BOC: %u\n", Command);
		return (0);
	}
	if (trace_pixel) {
		printf ("  Character code %lu:\n", Char_Code);
		printf ("    Backpointer to next character: %08lX\n", Temp);
		printf ("    Min_M, Max_M, Min_N, Max_N: %ld, %ld, %ld, %ld\n", Char_Bounds[0],
			Char_Bounds[1], Char_Bounds[2], Char_Bounds[3]);
	}
/*
 *	Start a character by allocating scratch memory for storing the
 *	raster, initial setting of row and column within raster:
 */
	Width = Char_Bounds[1] - Char_Bounds[0] + 1;
	Temp = Width * (Char_Bounds[3] - Char_Bounds[2] + 1);
	Cur_Ptr = Raster_Ptr = (unsigned char *) Mem_Alloc (Temp);
	for (; Temp > 0; Temp--)
		*Cur_Ptr++ = 0;
	Cur_Ptr = Row_Ptr = Raster_Ptr;
	Paint_Switch = 0;
/*
 *	Read through the file sequentially, up to the end of character, and
 *	process each character description:
 */
	while ((Command = Read_Character_M (GF_File)) != EOC && File_At_EOF_M (GF_File) == 0)
	switch (Command) {

	case_64(PAINT_0)
		Temp = Command - PAINT_0;
		goto Do_Paint;

	case_3(PAINT1)
		Temp = Read_Unsigned_Int_M (Command-PAINT1+1, GF_File);
Do_Paint:	if (Paint_Switch == 0)
			Cur_Ptr = &Cur_Ptr[Temp];
		else for (; Temp > 0; Temp--)
			*Cur_Ptr++ = 1;
		Paint_Switch ^= 1;
		break;

	case_4(SKIP0)
		Temp = Read_Unsigned_Int_M (Command-SKIP0, GF_File) + 1;
		Cur_Ptr = Row_Ptr = &Row_Ptr[Width*Temp];
		Paint_Switch = 0;
		break;

	case_165(NEW_ROW_0)
		Row_Ptr = &Row_Ptr[Width];
		Cur_Ptr = &Row_Ptr[Command-NEW_ROW_0];
		Paint_Switch = 1;
		break;

	case_4(XXX1)
		Skip (Read_Unsigned_Int_M (Command-XXX1+1, GF_File), GF_File);
		break;

	case_1(YYY)
		Read_Unsigned_Int_M (4, GF_File);
		break;

	case_1(NO_OP)
		break;

	default:
		if (trace_pixel)
			printf ("    Unexpected command while processing character: %u\n", Command);
		goto Out;
	}
/*
 *	End of character. Allocate the character descriptor for this
 *	character and fill in the relevant information:
 */
Out:	if (Command == EOC) {
		Reduce_Raster_M (Raster_Ptr, Width, Char_Bounds[3] - Char_Bounds[2] + 1,
			         -Char_Bounds[0], Char_Bounds[3], &Wid, &Hgt, &X0, &Y0);
		Char_Ptr = (struct Char_Definition *)
				Mem_Alloc (sizeof (struct Char_Definition) + ((Wid + 7) >> 3) * Hgt);
		Cur_Ptr = (unsigned char *) Char_Ptr;
		Char_Ptr->Driver_Id = (unsigned long) Font_Ptr->Font_Directory[Char_Code];
		Char_Ptr->DVI_Width = GF_Char_Info[Char_Code].Char_Width;
		Char_Ptr->H_Escapement = GF_Char_Info[Char_Code].H_Esc;
		Char_Ptr->V_Escapement = GF_Char_Info[Char_Code].V_Esc;
		Char_Ptr->Character_Code = Char_Code;
		Char_Ptr->Pixel_Width = Wid;
		Char_Ptr->Pixel_Height = Hgt;
		Char_Ptr->X_Origin = X0;
		Char_Ptr->Y_Origin = Y0;
		Char_Ptr->Pixel_Array = &Cur_Ptr[sizeof (struct Char_Definition)];
		Pack_Raster_Data_M (Wid, Hgt, Raster_Ptr, Char_Ptr->Pixel_Array);
		Font_Ptr->Font_Directory[Char_Code] = Char_Ptr;
		Char_Mask[Char_Code] = 1;
	}
	Mem_Free (Raster_Ptr);
	return ((Command == EOC) ? 1 : 0);
}
