/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains basic graphics procedures.
 */

#include "types.h"

#include "arith.p"

/*
 *	Function 'Clip_Line' clips a line segment to the limits of a
 *	rectangular clipping region. The function value it returns
 *	indicates whether the clipped line segment is within the
 *	rectangular region. The input line coordinates are replaced
 *	by the clipped line coordinates. If the function returns 0,
 *	the input line endpoints that remain are undefined.
 *
 *	X coordinates go from left to right, y coordinates go from
 *	top to bottom (that is, top <= bottom).
 *
 *	The algorithm used is the standard Cohen-Sutherland algorithm
 *	for 2-dimensional clipping. The primary weakness in most
 *	published version(s) of this algorithm is that the round-off
 *	effects that can occur when computing the intersections of the
 *	line with a boundary are ignored. This can cause the test/intersect
 *	cycle to oscillate back and forth, resulting in a non-terminating
 *	algorithm. This condition has to be carefully accounted for by
 *	noting that the line will intersect each boundary at most once.
 *	We are especially vulnerable to this problem here, since all
 *	arithmetic is done using integers, not reals.
 */

int Clip_Line (p1, p2, Left, Right, Top, Bottom)
long p1[2], p2[2], Left, Right, Top, Bottom;
{
	auto   long X_Slope, Y_Slope, Temp;
	auto   unsigned int c1, c2, Done;
	extern unsigned int Code();

	c1 = Code (p1[0], p1[1], Left, Right, Top, Bottom);
	c2 = Code (p2[0], p2[1], Left, Right, Top, Bottom);
	X_Slope = p2[0] - p1[0];
	Y_Slope = p2[1] - p1[1];
	Done = 0;
	while ((c1 | c2) != 0) {
		if ((c1 & c2) != 0)	/* Line is completely outside bounds */
			return (0);
		if (c1 == 0) {		/* Swap endpoints so p1 is outside */
			Temp = p1[0]; p1[0] = p2[0]; p2[0] = Temp;
			Temp = p1[1]; p1[1] = p2[1]; p2[1] = Temp;
			X_Slope = -X_Slope; Y_Slope = -Y_Slope;
			c1 = c2; c2 = 0;
		}
		if ((c1 & 0x01) != 0) {		/* Crosses left edge */
			p1[1] += XN_Div_D_R_M (Left - p1[0], Y_Slope, X_Slope);
			p1[0] = Left;
			Done |= 0x01;
		} else if ((c1 & 0x02) != 0) {	/* Crosses right edge */
			p1[1] += XN_Div_D_R_M (Right - p1[0], Y_Slope, X_Slope);
			p1[0] = Right;
			Done |= 0x02;
		} else if ((c1 & 0x04) != 0) {	/* Crosses top edge */
			p1[0] += XN_Div_D_R_M (Top - p1[1], X_Slope, Y_Slope);
			p1[1] = Top;
			Done |= 0x04;
		} else if ((c1 & 0x08) != 0) {	/* Crosses bottom edge */
			p1[0] += XN_Div_D_R_M (Bottom - p1[1], X_Slope, Y_Slope);
			p1[1] = Bottom;
			Done |= 0x08;
		}
		c1 = Code (p1[0], p1[1], Left, Right, Top, Bottom) & ~Done;
	}
/*
 *	Finally, compensate for rounding effects:
 */
	if (p1[0] < Left)
		p1[0] = Left;
	else if (p1[0] > Right)
		p1[0] = Right;
	if (p1[1] < Top)
		p1[1] = Top;
	else if (p1[1] > Bottom)
		p1[1] = Bottom;
	if (p2[0] < Left)
		p2[0] = Left;
	else if (p2[0] > Right)
		p2[0] = Right;
	if (p2[1] < Top)
		p2[1] = Top;
	else if (p2[1] > Bottom)
		p2[1] = Bottom;
	return (1);
}

unsigned int Code (x, y, Left, Right, Top, Bottom)
long x, y, Left, Right, Top, Bottom;
{
	auto   unsigned int c;

	c = 0;
	if (x < Left)
		c |= 0x01;
	else if (x > Right)
		c |= 0x02;
	if (y < Top)
		c |= 0x04;
	else if (y > Bottom)
		c |= 0x08;
	return (c);
}

/*
 *	Routine Clip_Arc clips an arc segment into at most four individual
 *	segments.
 */

unsigned int Clip_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Left, Right, Top, Bottom, Start_List, Stop_List)
long X, Y, Start_Ang, Stop_Ang, Start_List[4], Stop_List[4];
long Left, Right, Top, Bottom;
unsigned long Radius;
{
/* *** stubbed for now *** */
	Start_List[0] = Start_Ang;
	Stop_List[0] = Stop_Ang;
	return (1);
}

/*
 *	Routine Clip_Polygon clips a polygon to the specified bounds.
 */

unsigned int Clip_Polygon (Vertex_Count, Poly_X, Poly_Y, Left, Right, Top, Bottom, Clipped_X, Clipped_Y)
unsigned int Vertex_Count;
long Poly_X[], Poly_Y[], Left, Right, Top, Bottom, Clipped_X[], Clipped_Y[];
{
	auto   unsigned int Index;
/* *** stubbed for now *** */
	for (Index = 0; Index < Vertex_Count; Index++) {
		Clipped_X[Index] = Poly_X[Index];
		Clipped_Y[Index] = Poly_Y[Index];
	}
	return (Vertex_Count);
}

/*
 *	Routine 'Clip_Rule' clips a rule to the specified bounds.
 *	The bounds are given in such a way such that:
 *
 *	    Left <= u < Right    and    Top <= v < Bottom
 *
 *	and the rule specification specifies that:
 *
 *	    x <= u < x + w       and    y - h < v <= y.
 *
 *	If the function returns a zero value, it means the entire
 *	rule is out-of-bounds.
 */

int Clip_Rule (xy, wh, Left, Right, Top, Bottom)
long xy[2], Left, Right, Top, Bottom;
unsigned long wh[2];
{
	if (xy[0] >= Right || xy[1] < Top || xy[0] + (long) wh[0] <= Left ||
	    xy[1] + 1 >= Bottom + (long) wh[1])
		return (0);
/*
 *	Clip x,w values:
 */
	if (xy[0] < Left) {
		wh[0] -= (unsigned long) (Left - xy[0]);
		xy[0] = Left;
	}
	if (xy[0] + (long) wh[0] > Right)
		wh[0] = (unsigned long) (Right - xy[0]);
/*
 *	Clip y,h values:
 */
	if (xy[1] >= Bottom) {
		wh[1] -= (unsigned long) (xy[1] - Bottom + 1);
		xy[1] = Bottom - 1;
	}
	if (xy[1] + 1 < Top + (long) wh[1])
		wh[1] = (unsigned long) (xy[1] - Top + 1);
	return (1);
}

/*
 *	Routine 'Segment_Line' takes a line segment, a dash-line
 *	specification and a starting phase, and breaks the line up into
 *	individual solid line segments. A function is supplied which
 *	is then called to draw the line. The phase of the next
 *	segment is returned as the function value.
 *
 *	The segment descriptor consists of an even number of length
 *	values, each describing the length of a segment. The first,
 *	third, etc. entries are solid segments, the second, fourth,
 *	etc. are for invisible segments. The line coordinates, width,
 *	phase and segment lengths must all be in the same units (usually
 *	1/1024 pixels).
 *
 *	When a visible segment is to be drawn, the supplied function
 *	is called with the endpoints of the line segment and the width
 *	of the line segment.
 */

unsigned long Segment_Line (X1, Y1, X2, Y2, Segment_Count, Segment_Desc, Phase, Width, Func)
long X1, Y1, X2, Y2;
unsigned int Segment_Count;
unsigned long Segment_Desc[], Phase, Width;
int (*Func)();
{
	auto   unsigned int Index, Visible;
	auto   unsigned long Current_Length, Line_Length, Segment_Length;
	auto   unsigned long Total_Phase;
	auto   long Delta_X, Delta_Y;
	extern double sqrt();
/*
 *	First, determine the total length of the segment descriptor
 *	(better not be zero):
 */
	Total_Phase = 0;
	for (Index = 0; Index < Segment_Count; Index++)
		Total_Phase += Segment_Desc[Index];
	if (Total_Phase == 0)
		return (Phase);
/*
 *	Cycle through the line segments to the starting phase:
 */
	Segment_Length = 0;
	Visible = 1;
	for (Index = 0; (Segment_Length += Segment_Desc[Index]) <= Phase; ) {
		Visible ^= 1;
		if (++Index == Segment_Count)
			Index = 0;
	}
/*
 *	Compute some preliminaries:
 */
	Delta_X = X2 - X1;
	Delta_Y = Y2 - Y1;
	Line_Length = (long) sqrt ((double) Delta_X * (double) Delta_X +
				   (double) Delta_Y * (double) Delta_Y);
	if (Line_Length == 0) {		/* Special case, to avoid divide by zero below */
		if (Visible != 0)
			(*Func) (X1, Y1, X1, Y1, Width);
		return (Phase);
	}
	Current_Length = 0;
	Segment_Length -= Phase;
/*
 *	Cycle through the segment array, drawing the solid segments:
 */
	for (;;) {
		if (Current_Length + Segment_Length > Line_Length)
			Segment_Length = Line_Length - Current_Length + 1;
		if (Segment_Length > 0 && Visible != 0)
			(*Func) (XN_Div_D_R_M (Current_Length, Delta_X, Line_Length) + X1,
				 XN_Div_D_R_M (Current_Length, Delta_Y, Line_Length) + Y1,
				 XN_Div_D_R_M (Current_Length + Segment_Length - 1, Delta_X, Line_Length) + X1,
				 XN_Div_D_R_M (Current_Length + Segment_Length - 1, Delta_Y, Line_Length) + Y1,
				 Width);
		if ((Current_Length += Segment_Length) >= Line_Length)
			break;
		if (++Index == Segment_Count)
			Index = 0;
		Segment_Length = Segment_Desc[Index];
		Visible ^= 1;
	}
	return ((Phase + Line_Length) % Total_Phase);
}

/*
 *	Routine Build_Line takes a solid line segment of specified
 *	width and draws it by drawing lesser-width line segments as
 *	required to build up the entire line's width. All dimensions
 *	are in 1/1024 pixels. Max_Width should be a multiple of 1024.
 */

Build_Line (X1, Y1, X2, Y2, Width, Max_Width, Func)
long X1, Y1, X2, Y2;
unsigned long Width, Max_Width;
int (*Func)();
{
	auto   unsigned long Line_Length, Width_Remaining, Line_Width, Dimen;
	auto   long Delta_X, Delta_Y, Dx, Dy;
	extern double sqrt();
/*
 *	First, draw the central line segment (line width must always
 *	be odd multiple of 1024):
 */
	Line_Width = (((Width < Max_Width) ? Width : Max_Width) + 1023) & ~0x3FF;
	if ((Line_Width & 0x0400) == 0)
		Line_Width -= 1024;
	(*Func) (X1, Y1, X2, Y2, Line_Width);
/*
 *	Now, draw segments on either side of the line segment
 *	already drawn until the required width is achieved:
 */
	Delta_X = X2 - X1;
	Delta_Y = Y2 - Y1;
	Line_Length = (long) sqrt ((double) Delta_X * (double) Delta_X +
				   (double) Delta_Y * (double) Delta_Y);
	if (Line_Length != 0) {
		Width_Remaining = (Width + 1023) & ~0x3FF;
		if ((Width_Remaining & 0x0400) == 0)
			Width_Remaining -= 1024;
		Dimen = Line_Width;
		while ((Width_Remaining -= Line_Width) > 0) {
			if ((Line_Width = Width_Remaining >> 1) > Max_Width)
				Line_Width = Max_Width;
			if ((Line_Width & 0x0400) == 0)
				Line_Width -= 1024;
			Dimen = (Dimen + Line_Width) >> 1;
			Dx = XN_Div_D_R_M (Dimen, Delta_Y, Line_Length);
			Dy = XN_Div_D_R_M (Dimen, Delta_X, Line_Length);
			(*Func) (X1 + Dx, Y1 - Dy, X2 + Dx, Y2 - Dy, Line_Width);
			(*Func) (X1 - Dx, Y1 + Dy, X2 - Dx, Y2 + Dy, Line_Width);
			Dimen += Dimen + Line_Width;
			Width_Remaining -= Line_Width;
		}
	}
}

/*
 *	Routine Segment_Arc performs the same function as Segment_Line,
 *	except it is for arc segments.
 */

unsigned long Segment_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Segment_Count, Segment_Desc, Phase, Width, Func)
long X, Y, Start_Ang, Stop_Ang;
unsigned int Segment_Count;
unsigned long Segment_Desc[], Phase, Width;
int (*Func)();
{
	auto   unsigned int Index, Visible;
	auto   unsigned long Current_Length, Arc_Length, Segment_Length;
	auto   unsigned long Total_Phase;
	auto   long Current_Angle, Angle;
	extern unsigned long Compute_Arc_Length(), Compute_Angle();
/*
 *	First, determine the total length of the segment descriptor
 *	(better not be zero):
 */
	Total_Phase = 0;
	for (Index = 0; Index < Segment_Count; Index++)
		Total_Phase += Segment_Desc[Index];
	if (Total_Phase == 0)
		return (Phase);
/*
 *	Cycle through the line segments to the starting phase:
 */
	Segment_Length = 0;
	Visible = 1;
	for (Index = 0; (Segment_Length += Segment_Desc[Index]) <= Phase; ) {
		Visible ^= 1;
		if (++Index == Segment_Count)
			Index = 0;
	}
/*
 *	Compute some preliminaries:
 */
	Arc_Length = Compute_Arc_Length (Radius, Stop_Ang - Start_Ang);
	if (Arc_Length == 0) {		/* Special case */
		if (Visible != 0)
			(*Func) (X, Y, Radius, Start_Ang, Stop_Ang, Width);
		return (Phase);
	}
	Current_Length = 0;
	Current_Angle = Start_Ang;
	Segment_Length -= Phase;
/*
 *	Cycle through the segment array, drawing the solid segments:
 */
	for (;;) {
		if (Current_Length + Segment_Length > Arc_Length)
			Segment_Length = Arc_Length - Current_Length + 1;
		Angle = Compute_Angle (Radius, Segment_Length);
		if (Stop_Ang < Start_Ang)
			Angle = -Angle;
		if (Segment_Length > 0 && Visible != 0)
			(*Func) (X, Y, Radius, Current_Angle, Current_Angle + Angle, Width);
		Current_Angle += Angle;
		if ((Current_Length += Segment_Length) >= Arc_Length)
			break;
		if (++Index == Segment_Count)
			Index = 0;
		Segment_Length = Segment_Desc[Index];
		Visible ^= 1;
	}
	return ((Phase + Arc_Length) % Total_Phase);
}

/*
 *	Routine Build_Arc performs the same function as Build_Line,
 *	except it is for arc segments.
 */

Build_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Width, Max_Width, Func)
long X, Y, Start_Ang, Stop_Ang;
unsigned long Radius, Width, Max_Width;
int (*Func)();
{
	auto   unsigned long Width_Remaining, Arc_Width, Dimen;
/*
 *	First, draw the central arc segment (arc width must always
 *	be odd multiple of 1024):
 */
	Arc_Width = (((Width < Max_Width) ? Width : Max_Width) + 1023) & ~0x3FF;
	if ((Arc_Width & 0x0400) == 0)
		Arc_Width -= 1024;
	(*Func) (X, Y, Radius, Start_Ang, Stop_Ang, Arc_Width);
/*
 *	Now, draw segments on either side of the arc segment
 *	already drawn until the required width is achieved:
 */
	Width_Remaining = (Width + 1023) & ~0x3FF;
	if ((Width_Remaining & 0x0400) == 0)
		Width_Remaining -= 1024;
	Dimen = Arc_Width;
	while ((Width_Remaining -= Arc_Width) > 0) {
		if ((Arc_Width = Width_Remaining >> 1) > Max_Width)
			Arc_Width = Max_Width;
		if ((Arc_Width & 0x0400) == 0)
			Arc_Width -= 1024;
		Dimen = (Dimen + Arc_Width) >> 1;
		(*Func) (X, Y, Radius + Dimen, Stop_Ang, Start_Ang, Arc_Width);
		(*Func) (X, Y, Radius - Dimen, Start_Ang, Stop_Ang, Arc_Width);
		Dimen += Dimen + Arc_Width;
		Width_Remaining -= Arc_Width;
	}
}

/*
 *	Routine Flatten_Arc converts a solid arc segment to a series
 *	of short line segments, to simulate the arc.
 */

Flatten_Arc (X, Y, Radius, Start_Ang, Stop_Ang, Width, Max_Arc_Length, Func)
long X, Y, Start_Ang, Stop_Ang;
unsigned long Radius, Width, Max_Arc_Length;
int (*Func)();
{
	auto   long Angle, Current_Angle, X0, Y0, X1, Y1;
	extern unsigned long Compute_Angle();

	Angle = Compute_Angle (Radius, Max_Arc_Length);
	if (Angle == 0)
		Angle = 1;
	if (Stop_Ang < Start_Ang)
		Angle = -Angle;
	Current_Angle = Start_Ang;
	Compute_Arc_XY (X, Y, Radius, Current_Angle, &X1, &Y1);
	if (Angle == 0) {
		(*Func) (X1, Y1, X1, Y1, Width);
		return;
	}
	for (;;) {
		X0 = X1; Y0 = Y1;
		Current_Angle += Angle;
		Compute_Arc_XY (X, Y, Radius, Current_Angle, &X1, &Y1);
		(*Func) (X0, Y0, X1, Y1, Width);
		if (Stop_Ang > Start_Ang) {
			if (Current_Angle >= Stop_Ang)
				break;
		} else
			if (Current_Angle <= Stop_Ang)
				break;
	}
}

/*
 *	Routine Compute_Arc_XY computes the (x,y) point on an arc
 *	segment endpoint.
 */

Compute_Arc_XY (X, Y, Radius, Angle, X_Ptr, Y_Ptr)
long X, Y, Angle, *X_Ptr, *Y_Ptr;
unsigned long Radius;
{
	auto   double Ang, Rad;
	extern double sin(), cos();

	Ang = (double) Angle * 0.0000174532925199432958;
	Rad = (double) Radius;
	*X_Ptr = X + (long) (Rad * cos (Ang) + 0.5);
	*Y_Ptr = Y + (long) (Rad * sin (Ang) + 0.5);
}

/*
 *	Routine Compute_Arc_Length computes the arc length of an
 *	arc segment. The Angle is in 1/1000 degrees.
 *
 *	The funny numbers that figure in the formulas below relate
 *	to the value of pi / 180000. Best ratio estimate of this
 *	is 36433 / 2087457135.
 */

unsigned long Compute_Arc_Length (Radius, Angle)
unsigned long Radius;
long Angle;
{
	return (XN_Div_D_R_M (Radius, XN_Div_D_R_M ((Angle >= 0) ? Angle : -Angle, 36433, 285), 7324411));
}

unsigned long Compute_Angle (Radius, Arc_Length)
unsigned long Radius, Arc_Length;
{
	return (XN_Div_D_R_M (XN_Div_D_R_M (Arc_Length, 7324411, Radius), 285, 36433));
}

/*
 *	Routine Fill_Circle draws a filled circle using a series of
 *	solid line segments. The pen width parameter should be small
 *	enough so that the rounded edge of the pen is not discernable
 *	on the circle's edge. It should be an odd multiple of 1024,
 *	and the radius should be a multiple of the pen width to obtain
 *	good results.
 */

Fill_Circle (X, Y, Radius, Pen_Width, Func)
long X, Y;
unsigned long Radius, Pen_Width;
int (*Func)();
{
	auto   double rr;
	auto   long a, b, Limit;
	auto   unsigned int Index;
	extern double sqrt();

	if ((Limit = Radius - (Pen_Width >> 1)) < 0)
		Limit = 0;
	(*Func) (X-Limit, Y, X+Limit, Y, Pen_Width);
	rr = (double) Limit; rr *= rr;
	a = 0;
	while ((a += Pen_Width) <= Limit) {
		b = (long) sqrt (rr - (double) a * (double) a);
		(*Func) (X+b, Y-a, X-b, Y-a, Pen_Width);
		(*Func) (X-b, Y+a, X+b, Y+a, Pen_Width);
	}
}
